/*
 * @(#)TrianglesU.c
 *
 * Copyright 1994 - 2023  David A. Bagley, bagleyd AT verizon.net
 *
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the author not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.
 *
 * This program is distributed in the hope that it will be "useful",
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */

/* Undo algorithm for Triangles */

#include "TrianglesP.h"

static int startSpace[MAX_ORIENT];
static int startRow[MAX_ORIENT][ROW_TYPES];
int *startPosition = NULL;

static void
newStack(TrianglesStack *s)
{
	if (s->lastMove != NULL || s->firstMove != NULL)
		return;
	if (!(s->lastMove = (MoveStack *) malloc(sizeof (MoveStack)))) {
		DISPLAY_ERROR("Not enough memory, exiting.");
	}
	if (!(s->firstMove = (MoveStack *) malloc(sizeof (MoveStack)))) {
		DISPLAY_ERROR("Not enough memory, exiting.");
	}
	s->firstMove->previous = s->lastMove->next = NULL;
	s->firstMove->next = s->lastMove;
	s->lastMove->previous = s->firstMove;
	s->count = 0;
}

static void
pushStack(TrianglesStack *s, MoveRecord **move)
{
	if (!(s->currMove = (MoveStack *) malloc(sizeof (MoveStack)))) {
		DISPLAY_ERROR("Not enough memory, exiting.");
	}
	s->lastMove->previous->next = s->currMove;
	s->currMove->previous = s->lastMove->previous;
	s->currMove->next = s->lastMove;
	s->lastMove->previous = s->currMove;
	*move = &(s->currMove->move);
	s->count++;
}

static void
popStack(TrianglesStack *s)
{
	s->currMove = s->lastMove->previous;
	s->lastMove->previous->previous->next = s->lastMove;
	s->lastMove->previous = s->lastMove->previous->previous;
	free(s->currMove);
	s->count--;
}


static MoveRecord *
topStack(TrianglesStack *s)
{
	return &(s->lastMove->previous->move);
}

static int
emptyStack(TrianglesStack *s)
{
	return (s->lastMove->previous == s->firstMove);
}

static void
flushStack(TrianglesStack *s)
{
	while (s->lastMove->previous != s->firstMove) {
		s->currMove = s->lastMove->previous;
		s->lastMove->previous->previous->next = s->lastMove;
		s->lastMove->previous = s->lastMove->previous->previous;
		free(s->currMove);
	}
	s->count = 0;
}

static void
deleteStack(TrianglesStack *s)
{
	flushStack(s);
	if (s->firstMove) {
		free(s->firstMove);
		s->firstMove = NULL;
	}
	if (s->lastMove) {
		free(s->lastMove);
		s->lastMove = NULL;
	}
}

/**********************************/

void
newMoves(TrianglesStack *s)
{
	newStack(s);
}

void
deleteMoves(TrianglesStack *s)
{
	deleteStack(s);
}

static void
writeMove(MoveRecord *move, int direction)
{
#if 0
	move->direction = direction;
#endif
	move->packed = (unsigned char) (direction & 0xF);
}

static void
readMove(MoveRecord *move, int *direction)
{
#if 0
	*direction = move->direction;
#endif
	*direction = (int) (move->packed & 0xF);
}

void
setMove(TrianglesStack *s, int direction)
{
	MoveRecord *move;

	pushStack(s, &move);
	writeMove(move, direction);
}

void
getMove(TrianglesStack *s, int *direction)
{
	readMove(topStack(s), direction);
	popStack(s);
}

int
madeMoves(TrianglesStack *s)
{
	return !emptyStack(s);
}

void
flushMoves(TrianglesWidget w, TrianglesStack *s, Boolean undo)
{
	int i;

	flushStack(s);
	if (undo) {
		startSpace[UP] = w->triangles.spacePosition[UP];
		startSpace[DOWN] = w->triangles.spacePosition[DOWN];
		startRow[UP][TRBL] = w->triangles.spaceRow[UP][TRBL];
		startRow[UP][TLBR] = w->triangles.spaceRow[UP][TLBR];
		startRow[UP][ROW] = w->triangles.spaceRow[UP][ROW];
		startRow[DOWN][TRBL] = w->triangles.spaceRow[DOWN][TRBL];
		startRow[DOWN][TLBR] = w->triangles.spaceRow[DOWN][TLBR];
		startRow[DOWN][ROW] = w->triangles.spaceRow[DOWN][ROW];
		for (i = 0; i < w->triangles.sizeSize; i++)
			startPosition[i] = w->triangles.tileOfPosition[i];
	}
}

int
numMoves(TrianglesStack *s)
{
	return s->count;
}

Boolean
scanMoves(FILE *fp, TrianglesWidget w, int moves)
{
	int direction, l, c;

	for (l = 0; l < moves; l++) {
		while ((c = getc(fp)) != EOF && c != SYMBOL);
		if (fscanf(fp, "%d", &direction) != 1) {
			(void) fprintf(stderr,
				"corrupt scan, expecting an integer for move %d\n", l);
			return False;
		}
		if (!movePuzzleDir(w, direction, INSTANT)) {
			(void) fprintf(stderr,
				"%d move in direction %d, cannot be made.",
				l, direction);
			return False;
		}
	}
	return True;
}

void
printMoves(FILE *fp, TrianglesStack *s)
{
	int direction, counter = 0;

	s->currMove = s->firstMove->next;
	(void) fprintf(fp, "moves\tdirection\n");
	while (s->currMove != s->lastMove) {
		readMove(&(s->currMove->move), &direction);
		(void) fprintf(fp, "%d%c\t%d\n", ++counter, SYMBOL, direction);
		s->currMove = s->currMove->next;
	}
}

Boolean
scanStartPosition(FILE *fp, TrianglesWidget w)
{
	int i, num, c;

	while ((c = getc(fp)) != EOF && c != SYMBOL);
	for (i = 0; i < w->triangles.sizeSize; i++) {
		if (fscanf(fp, "%d ", &num) != 1) {
			(void) fprintf(stderr,
				"corrupt scan, expecting an integer for %d\n", i);
			return False;
		}
		startPosition[i] = num;
		if (!num)
			startSpace[UP] = i;
		else if (num == -1)
			startSpace[DOWN] = i;
	}
	for (i = DOWN; i <= UP; i++) {
		startRow[i][ROW] = toRow(w, startSpace[i]);
		startRow[i][TRBL] = toTrBl(w, startSpace[i], startRow[i][ROW]);
		startRow[i][TLBR] = toTlBr(w, startSpace[i], startRow[i][ROW]);
	}
	return True;
}

void
printStartPosition(FILE *fp, TrianglesWidget w)
{
	int r = 0, oldR = -1, p, space, temp;

	(void) fprintf(fp, "\nstartingPosition%c\n", SYMBOL);
	for (p = 0; p < w->triangles.sizeSize; p++) {
		if (oldR != r) {
			for (space = 0; space < w->triangles.sizeY - r - 1; space++)
				(void) fprintf(fp, "    ");
			oldR = r;
		}
		(void) fprintf(fp, "%4d", startPosition[p]);
		temp = (r + 1) * (r + 1)
			+ 2 * (r + 1) * (w->triangles.sizeX - 1) - 1;
		if (p == temp) {
			(void) fprintf(fp, "\n");
			r++;
		}
	}
	(void) fprintf(fp, "\n");
}

void
setStartPosition(TrianglesWidget w)
{
	int i;

	w->triangles.spacePosition[UP] = startSpace[UP];
	w->triangles.spacePosition[DOWN] = startSpace[DOWN];
	w->triangles.spaceRow[UP][TRBL] = startRow[UP][TRBL];
	w->triangles.spaceRow[UP][TLBR] = startRow[UP][TLBR];
	w->triangles.spaceRow[UP][ROW] = startRow[UP][ROW];
	w->triangles.spaceRow[DOWN][TRBL] = startRow[DOWN][TRBL];
	w->triangles.spaceRow[DOWN][TLBR] = startRow[DOWN][TLBR];
	w->triangles.spaceRow[DOWN][ROW] = startRow[DOWN][ROW];
	for (i = 0; i < w->triangles.sizeSize; i++)
		w->triangles.tileOfPosition[i] = startPosition[i];
	drawAllTiles(w);
}
